# %%
import os
import fabio_functions
import numpy as np
import pandas as pd
from pathlib import Path
import importlib

# %%
years = np.arange(1995, 2021)

fabio_path = None  # Insert path to FABIO 
exiobase_path = None  # Insert path to EXIOBASE
save_folder = None  # Insert path to save folder

# %%
# Load and prepare mappings
regions_df = pd.read_csv(
    "./data/fabio-exio_reg.csv",
    index_col=[0]
)

fabio_exio_supply_df = (
    pd.read_csv(
        "./data/fabio-exio_sup_v1.2.csv",
        header=[0],
        index_col=[0, 1, 2, 3]
    )
    .replace(0, np.nan)
    .rename_axis(["sector"], axis=1)
    .droplevel([0, 2, 3])
    .rename_axis(["item"], axis=0)
    .stack()
    .reset_index()
    .drop([0], axis=1)
)

# %%
for year in years:
    reader = fabio_functions.read(
        fabio_path / "biomass",
        year=year,
        version=1.2
    )
    Ef = reader.E()
    
    F_landuse = (
        Ef
        .drop([
            "area_code",
            "item_code",
            "comm_code",
            "comm_group",
            "group",
            "biomass",
            "blue",
            "green",
            "p_application",
            "n_application"
        ], axis=1)
    )
            
    
    MRIOT_path = (
        exiobase_path
        / f"IOT_{year}_pxp"
    )
    
    Fe = pd.read_csv(
        MRIOT_path / "satellite" / "F.txt",
        sep="\t",
        index_col=[0],
        header=[0, 1]
    )
    
    crop_landuse_sectors = [
        'Cropland - Cereal grains nec',
        'Cropland - Crops nec',
        'Cropland - Fodder crops-Cattle',
        'Cropland - Fodder crops-Meat animals nec',
        'Cropland - Fodder crops-Pigs',
        'Cropland - Fodder crops-Poultry',
        'Cropland - Fodder crops-Raw milk',
        'Cropland - Oil seeds',
        'Cropland - Paddy rice',
        'Cropland - Plant-based fibers',
        'Cropland - Sugar cane, sugar beet',
        'Cropland - Vegetables, fruit, nuts',
        'Cropland - Wheat',
    ]
    other_landuse_sectors = [
        'Infrastructure land',
        'Other land - Marginal use',
        'Other land - Wood fuel',
    ]
    grazing_landuse_sectors = [
        'Other land - Grazing',
        'Other land - Grazing-Cattle',
        'Other land - Grazing-Meat animals nec',
        'Other land - Grazing-Raw milk'
    ]
    E_landuse = (
        Fe
        .loc[
            crop_landuse_sectors + other_landuse_sectors + grazing_landuse_sectors,
            Fe.columns.isin(list(set(fabio_exio_supply_df["sector"])), level="sector")
        ]
        .unstack()
        .to_frame("EXIOBASE estimate")
        .reset_index()
    )
    
    df = (
        F_landuse.rename({"landuse": "FABIO estimate"}, axis=1)
        .merge(
            fabio_exio_supply_df,
            on="item",
            how="left"
        )
        .merge(
            regions_df.loc[:, ["area", "region"]],
            on="area",
            how="left"
        )
    )
    
    df_grazing = df[df["item"].isin(["Grazing"])]
    df_grazing["FABIO estimate agg"] = (
        df_grazing
        .groupby(["region"])["FABIO estimate"].transform("sum")
    )
    df_grazing["FABIO shares"] = df_grazing.loc[:, "FABIO estimate"].div(df_grazing.loc[:, "FABIO estimate agg"])
    df_grazing = (
        df_grazing
        .merge(
            (
                E_landuse[E_landuse["stressor"].isin(grazing_landuse_sectors)]
                .groupby(["region", "stressor"])["EXIOBASE estimate"].sum()
                .to_frame("EXIOBASE estimate")
                .reset_index()
            ),
            on=["region"],
            how="left"
        )
    )
    df_grazing["land use"] = df_grazing["FABIO shares"].mul(df_grazing["EXIOBASE estimate"])
    
    df_cropland = df[~df["item"].isin(["Grazing"])]
    df_cropland["FABIO estimate agg"] = (
        df_cropland
        .groupby(["region", "sector"])["FABIO estimate"].transform("sum")
    )
    df_cropland["FABIO shares"] = df_cropland.loc[:, "FABIO estimate"].div(df_cropland.loc[:, "FABIO estimate agg"])
    df_cropland = (
        df_cropland
        .merge(
            (
                E_landuse[~E_landuse["stressor"].isin(grazing_landuse_sectors)]
            ),
            on=["region", "sector"],
            how="left"
        )
    )
    df_cropland["land use"] = df_cropland["FABIO shares"].mul(df_cropland["EXIOBASE estimate"])
    
    leftover = (
        df_cropland
        .loc[
            (df_cropland["EXIOBASE estimate"] > 0)
            & (df_cropland["FABIO estimate agg"]==0)
            ,
            :
        ]
    )
    
    df_foddercrops = df[df["item"].isin(["Fodder crops"])]
    
    df_foddercrops["FABIO estimate agg"] = (
        df_foddercrops
        .groupby(["region", "sector"])["FABIO estimate"].transform("sum")
    )
    df_foddercrops["FABIO shares"] = df_foddercrops.loc[:, "FABIO estimate"].div(df_foddercrops.loc[:, "FABIO estimate agg"])
    df_foddercrops = (
        df_foddercrops
        .merge(
            leftover.drop_duplicates(subset=["area", "sector", "region"]).loc[:, ["area", "stressor", "EXIOBASE estimate"]],
            on=["area"],
            how="left"
        )
    )
    df_foddercrops["land use"] = df_foddercrops["FABIO shares"].mul(df_foddercrops["EXIOBASE estimate"]) 
    
    new_landuse = (
        pd.concat([
            df_foddercrops,
            df_grazing,
            df_cropland
        ], axis=0
        )
        .groupby(["area", "item", "stressor"])["land use"].sum()
        .unstack(["area", "item"]).replace(np.nan, 0)
    )
    save_path = (
        save_folder
        / f"{year}"
    )
    save_path.mkdir(parents=True, exist_ok=True)
    new_landuse.to_csv(save_path / "F.tsv", sep="\t")
    

